/*++

Copyright (c) 2005 AileronOS developers group
All rights reserved

File:

    dllman.c

Abstract:

    Used for DLL linkage when loading operating system

Author:

    11-Aug-2005          Vlad Maslakov

--*/

#include "include\dllman.h"
#include "include\cdefs.h"
#include "include\stdfunc.h"

#define TABLE_BASE 0x10000          // base address of the table of libraries
#define FNAME_LENGTH 11             // length of the file name
#define SCREEN_X 80                 // screen width
#define SCREEN_Y 25                 // screen height
#define VIDEOBUF 0xB8000            // video buffer

DWORD DLLs = 0;
BYTE LastError = 0;
int CursorPos = 0;
int Attribute = 7;

/*++

Routine description:

    Fills structure that describes current DLL file and it's
    position in memory

Arguments:

    Name - pointer to the valid name of the DLL library. It must
        contain 11 symbols including extension and file name

    Address - address of the library in memory

Returns:

    Next DLL file position or zero if error occured

--*/
DWORD STDCALL ProcessFile(char * Name, DWORD Address)
{
    PDWORD SigAddrPtr;          // pointer to the PE signature offset
    PWORD SigAddr;              // pointer to the PE signature
    DWORD PEBase;               // base address of PE header
    DWORD ObjBase;              // object table base
    WORD ObjCount;              // objects count
    DLL_LIBRARY Library;        // new library table element
    PDLL_LIBRARY MemDest;       // library element in memory
    int SecN;                   // number of the section
    int Cnt;                    // counter for several 'for' procedures
    BYTE NamePos;               // used durning reformatting library name
    PE_OBJECT CurObject;        // current object
    DWORD ImageSize;            // size of the image (return value)

    DWORD Foo;

    LPrintLine("ProcessFile Called\n");

    //
    // checking whether it a library
    //
    if (*(PWORD)Address != 0x5A4D) return 0;
    if (*(PWORD)(Address + 0x18) < 0x40) return 0;

    SigAddrPtr = Address + 0x3C;
    SigAddr = *SigAddrPtr + Address;
    if (*SigAddr != 0x4550) return 0;

    //
    // creating new library table element
    //
    PEBase = (DWORD)SigAddr;

    LPrintLine(Name);
    LPrintLine("\n");

    memcpy(Library.Name, Name, FNAME_LENGTH);

    Library.Name[11] = '\0';
    Library.ImageBase = Address;
    Library.ImpTable = *(PDWORD)(PEBase + 0x80) + Address;
    Library.ImpTableSize = *(PDWORD)(PEBase + 0x84);
    Library.ExpTable = *(PDWORD)(PEBase + 0x78) + Address;
    Library.ExpTableSize = *(PDWORD)(PEBase + 0x7C);
    Library.Entry = *(PDWORD)(PEBase + 0x28) + Address;

    //
    // make library name from name----ext format to name.ext
    //
    NamePos = 0;
    for (Cnt = 0; Cnt < 12; Cnt++)
        if (Library.Name[Cnt] != ' ')
        {
            if (Cnt == 8)
            {
                Library.Name[NamePos] = '.';
                NamePos++;
            }
            Library.Name[NamePos] = Library.Name[Cnt];
            NamePos++;
        }

    for (Cnt = NamePos; Cnt < 13; Cnt++)
        Library.Name[Cnt] = '\0';

    LToLowerCase(Library.Name);

/*    __asm__ ("mov dword ptr DLLs,esp");
    LPrintInt(DLLs);
    __asm__ ("hlt"::);*/

    LPrintLine("Library name: ");

    LPrintLine(Library.Name);
    LPrintLine("\n");

    LPrintInt(PEBase);
    LPrintLine("\n");

    //
    // moving sections to the corresponding positions
    //
    ObjCount = *(PWORD)(PEBase + 0x6);
    ObjBase = PEBase + *(PWORD)(PEBase + 0x14) + 0x18;

    for (SecN = ObjCount - 1; SecN >= 0; SecN--)
    {
        CurObject = *(PPE_OBJECT)(ObjBase + SecN * sizeof(PE_OBJECT));
        memcpy(CurObject.SectionRVA + Address, CurObject.PhysicalOffset + Address,
            CurObject.PhysicalSize);
    }

    //
    // record element to the table
    //
    MemDest = TABLE_BASE + sizeof(DLL_LIBRARY) * DLLs;
    *MemDest = Library;
    DLLs++;

    //
    // return the image size
    //
    ImageSize = *(PDWORD)(PEBase + 0x50);
    LPrintInt(ImageSize);
    LPrintLine("\n");

    return ImageSize + Address;
}

/*++

Routine description:

    Links the procedures in each library to procedures in others

--*/
void STDCALL PerformLinkage()
{
    BYTE Done;
    DLL_LIBRARY ActiveLib;

    LPrintLine("PerformLinkage called\n");

    for (Done = 0; Done < DLLs; Done++)
    {
        ActiveLib = *(PDLL_LIBRARY)(TABLE_BASE + sizeof(DLL_LIBRARY) * Done);
        LPrintLine("IMPS1 ");
        LPrintInt(ActiveLib.ImpTableSize);
        LPrintLine("\n");
        LinkLibrary(ActiveLib);
    }
}

/*++

Routine description:

    Binds a library to others

Arguments:

    Library - DLL library descriptor

Returns:

    Success of the operation

--*/
BOOL STDCALL LinkLibrary(DLL_LIBRARY Library)
{
    BYTE Entries;               // import table entries count
    int EntryN;                 // number of current entry
    int ImpN;                   // import procedure number
    IMPORT_ENTRY Entry;         // entry of the import table
    DWORD LookupRVA;            // lookup table RVA
    char SrcLibName[12];        // source library name
    EXPORT_TABLE CurExp;        // export table
    DLL_LIBRARY SrcLib;         // souce DLL library
    DWORD CLookupRVA;           // current lookup table RVA
    char ServiceName[32];       // name of external procedure

    LPrintLine("LinkLibrary called\n");

/*    LPrintLine("IMPS2 ");
    LPrintInt(Library.ImpTableSize);
    LPrintLine("\n");*/

    Entries = Library.ImpTableSize / sizeof(IMPORT_ENTRY);

    LPrintLine("Entries # ");
    LPrintInt(Entries);
    LPrintLine("\n");

    for (EntryN = 0; EntryN < Entries; EntryN++)
    {
        //
        // get information about current procedure needed to import and about
        // a source library
        //
        Entry = *(PIMPORT_ENTRY)(Library.ImpTable + sizeof(IMPORT_ENTRY) * EntryN);
        LookupRVA = *(PDWORD)(Entry.LookupRVA + Library.ImageBase);
        strcpy(SrcLibName, Library.ImageBase + Entry.NameRVA);

        //
        // find a source library
        //
        SrcLib = GetExpByName(SrcLibName);
        if (LastError != 0) return FALSE;
        CurExp = *(PEXPORT_TABLE)(SrcLib.ExpTable);

        //
        // link procedures
        //
        ImpN = 0;
        CLookupRVA = *(PDWORD)(LookupRVA + Library.ImageBase);        
        while (CLookupRVA != 0)
        {
            strcpy(ServiceName, CLookupRVA + sizeof(WORD) + Library.ImageBase);

            *(PDWORD)(Library.ImageBase + Entry.AdTabRVA + sizeof(DWORD) * ImpN) =
                GetProcAddress(SrcLib, CurExp, ServiceName);

            if (LastError != 0) return FALSE;

            ImpN++;
            CLookupRVA = *(PDWORD)(LookupRVA + Library.ImageBase);
        }
    }

    LPrintLine("Linkage for this DLL finished");

    return TRUE;
}

/*++

Routine description:

    Gets library export table by library's name

Arguments:

    Name - pointer to the library name

Returns:

    DLL library descriptor

--*/
DLL_LIBRARY STDCALL GetExpByName(char * Name)
{
    EXPORT_TABLE CurTable;          // current table
    BOOL Complete;                  // is search complete
    int CurEntry;                   // current DLL file
    DLL_LIBRARY CurDLL;             // current DLL file
    char * CurName;                 // current name of DLL

    Complete = FALSE;
    CurEntry = 0;

    while (CurEntry < DLLs && Complete == FALSE)
    {
        CurDLL = *(PDLL_LIBRARY)(TABLE_BASE + CurEntry * sizeof(DLL_LIBRARY));
        CurTable = *(PEXPORT_TABLE)CurDLL.ExpTable;
        CurName = CurTable.NameRVA + CurDLL.ImageBase;
        if (strcmp(Name, CurName) == 0)
            Complete = TRUE;

        DLLs++;
    }

    if (Complete == FALSE) LastError = 1;
    else LastError = 0;

    return CurDLL;
}

/*++

Routine description:

    Gets absolute address of the external procedure in memory

Arguments:

    Library - DLL library descriptor

    ExpTable - export table of the library that contains
        required procedure

    SvcName - name of the external procedure

Returns:

    Position of the procedure in memory or 0 if error
        occured

--*/
DWORD STDCALL GetProcAddress(DLL_LIBRARY Library, EXPORT_TABLE ExpTable, char * SvcName)
{
    char * NamePtr;             // pointer to the name of function
    BOOL Complete;              // is search procedure complete
    int CurName;                // index of the name
    DWORD PosInMem;             // result

    CurName = 0;
    Complete = FALSE;
    NamePtr = ExpTable.NamePtrs + Library.ImageBase;
    PosInMem = 0;

    while (Complete == FALSE && CurName < ExpTable.NamePtrs)
    {
        if (strcmp(SvcName, NamePtr) != 0)
        {
            CurName++;
            NamePtr = NamePtr + strlen(NamePtr) + 1;
        }
        else
        {
            PosInMem = *(PDWORD)(ExpTable.AddrTableRVA + Library.ImageBase + CurName * sizeof(DWORD)) +
                Library.ImageBase;
            Complete = TRUE;
        }
    }

    if (Complete == FALSE) LastError = 1;
    else LastError = 0;

    return PosInMem;
}

void STDCALL LPrintLine(char * Text)
{
    while (*Text)
    {
        LPrintChar(*Text);
        Text++;
    }
}

void STDCALL LPrintChar(char c)
{
    char * VideoBuf = VIDEOBUF;
    int Cnt;

    switch (c)
    {
    case '\n':
        CursorPos += SCREEN_X;
        CursorPos -= CursorPos % SCREEN_X;
        break;

    default:
        *(VideoBuf + CursorPos * 2) = c;
        *(VideoBuf + CursorPos * 2 + 1) = Attribute;
        CursorPos++;
        break;
    }

    if (CursorPos > SCREEN_X * SCREEN_Y)
    {
        for (Cnt = SCREEN_X * 2; Cnt <= SCREEN_X * SCREEN_Y * 2 + SCREEN_X * 2; Cnt++)
            *(VideoBuf + Cnt - SCREEN_X * 2) = *(VideoBuf + Cnt);

        CursorPos -= SCREEN_X;
    }
}

void STDCALL LToLowerCase(char * Text)
{
    while (*Text)
    {
        if (*Text >= 65 && *Text <= 90)
            *Text = *Text + 32;

        Text++;
    }
}

void STDCALL LPrintInt(DWORD Num)
{
    DWORD Divider;
    DWORD Result;
    int Cnt;

    Divider = 0x10000000;
    LPrintLine("0x");

    for (Cnt = 0; Cnt < 8; Cnt++)
    {
        Result = Num / Divider;
        Num = Num % Divider;
        Divider = Divider / 0x10;

        if (Result < 10) LPrintChar(48 + Result);
        else LPrintChar(55 + Result);
    }
}

